#!/usr/bin/env python3
# Compute max|I2| drift for the Casimir operator Q over admissible transforms.
# Acceptance: max|I2| drift <= 1e-3.
#
# HOW TO USE:
#   python check_i2_drift.py --coeffs_csv results\casimir_coeffs.csv --trials 2000 --seed 20250819
#
# If you have explicit O_a operators, implement build_Q_from_coeffs(B).
# Otherwise this script falls back to the toy baseline Q = diag(1, -1, 1, -1).

import json
import argparse
from pathlib import Path

import numpy as np


def second_elem_sym_poly(eigs):
    # I2 = sum_{i<j} λ_i λ_j
    s = 0.0
    n = len(eigs)
    for i in range(n):
        for j in range(i + 1, n):
            s += eigs[i] * eigs[j]
    return float(s)


def random_orthogonal(n, rng):
    # QR with random normal entries -> orthogonal with det ±1
    A = rng.normal(size=(n, n))
    Q, R = np.linalg.qr(A)
    d = np.sign(np.diag(R))
    Q = Q * d
    return Q


def build_Q_from_coeffs(B):
    """
    TODO (optional): Implement using your project’s operator definitions O_a.
    Given coeffs B={'B1':..., 'B2':..., 'B3':..., 'B4':...},
    return the 4x4 numpy array Q = sum_a B_a O_a^2 with A=0.
    """
    raise NotImplementedError("Define O_a and build Q here if desired.")


def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--coeffs_csv", default="results/casimir_coeffs.csv")
    ap.add_argument("--trials", type=int, default=200)
    ap.add_argument("--seed", type=int, default=20250819)
    ap.add_argument("--out", default="results/i2_drift.json")
    args = ap.parse_args()

    # Load coefficients (not used by fallback, but parsed for parity)
    import csv
    B = {}
    with open(args.coeffs_csv, newline="") as f:
        for row in csv.DictReader(f):
            B[row["coeff"]] = float(row["value"])

    # Build Q
    try:
        Q = build_Q_from_coeffs(B)
    except NotImplementedError:
        # Toy-model baseline used in V1 acceptance
        Q = np.diag([1.0, -1.0, 1.0, -1.0])

    eigs_ref = np.linalg.eigvalsh(Q)
    I2_ref = second_elem_sym_poly(eigs_ref)

    rng = np.random.default_rng(args.seed)
    drifts = []
    for _ in range(args.trials):
        O = random_orthogonal(Q.shape[0], rng)
        Qp = O.T @ Q @ O
        eigs = np.linalg.eigvalsh(Qp)
        I2 = second_elem_sym_poly(eigs)
        drifts.append(abs(I2 - I2_ref))

    result = {
        "I2_ref": I2_ref,
        "max_abs_drift": float(np.max(drifts)),
        "mean_abs_drift": float(np.mean(drifts)),
        "trials": args.trials,
        "seed": args.seed,
        "acceptance_threshold": 1e-3,
        "pass": bool(np.max(drifts) <= 1e-3),
    }
    Path("results").mkdir(exist_ok=True)
    with open(args.out, "w") as f:
        json.dump(result, f, indent=2)
    print(json.dumps(result, indent=2))


if __name__ == "__main__":
    main()
